# cracker-jokes for Pimoroni Badger 2040
# Shows Christmas cracker jokes on a wearable e-ink display
# Perfect for Christmas parties where you can't think up small talk
# Public Domain Andrew Oakley 2022-12 aoakley.com
# apart from the bits nicked from badger_os (C) Pimoroni.com probably
#
# I recommend a 2xCR2023 battery holder with on/off switch
# it will last at least 6 hours and probably several days
# Pimoroni part number ADA783
# Use hot glue or glue dots to affix to rear of badger
#
# Switch battery holder ON then press button A to start
# Press button B then switch battery holder OFF to stop
# Up/Down buttons to skip through previous/next joke in sequence
# Button A to move to random joke
# Button C to display wildly inaccurate battery percentage
#
# I learned to program aged 5 in 1976. This could be written much
# better using OO & events. But I'm old and don't think like that.
# Perfect is the enemy of good. Good is the enemy of "done".

# Import the libraries
import badger2040, time, math, random, sys, machine
# Set the processor speed to slow to save battery
badger2040.system_speed(badger2040.SYSTEM_SLOW)
# Mushroom, snake
badger = badger2040.Badger2040()

# White on black is easier to read from a distance IMHO
backgroundcolour=0     # 0=Black
foregroundcolour=15    # 15=White
callsecs=7             # Delay for set-up line of joke
responsesecs=5         # Delay for punchline
row=0                  # Position in joke file
laststartjoke=0        # Position of 1st line of previous joke
firstrun=True          # Whether this is the first run
width=badger2040.WIDTH # Width of display
# We're assuming the default font is about 30 pixels high
# I suppose that might change in future firmware
# If that ever happens, you'll need to specify the size here
heightfactor=30        # Pixel height between text
originx=3              # Offset from left
originy=-12            # Offset of the vertical centre of the text
badger.update_speed(badger2040.UPDATE_FAST)
badger.thickness(3)

def get_battery_level():
    # Battery measurement. This is VERY approximate.
    # I've assumed 2x3v CR2032 batteries = 6 volts full charge
    # Do NOT rely on this. You have been warned.
    # Will return <5% e.g. 0 if USB-C power connected
    # Stolen from badger_os , copyright Pimoroni probably
    vbat_adc = machine.ADC(badger2040.PIN_BATTERY)
    vref_adc = machine.ADC(badger2040.PIN_1V2_REF)
    vref_en = machine.Pin(badger2040.PIN_VREF_POWER)
    vref_en.init(machine.Pin.OUT)
    vref_en.value(0)
    
    # Enable the onboard voltage reference
    vref_en.value(1)
    # A moment to get a stable reading
    time.sleep(0.25)

    # Calculate the logic supply voltage, as will be lower that the usual 3.3V when running off low batteries
    vdd = 1.24 * (65535 / vref_adc.read_u16())
    vbat = (
        (vbat_adc.read_u16() / 65535) * 3 * vdd
    )  # 3 in this is a gain, not rounding of 3.3V

    # Disable the onboard voltage reference
    vref_en.value(0)

    # Convert the voltage to a percentage
    # 2.7v is minimum operating voltage (=0%, empty)
    # =>6v (2.7+3.3) is maximum (=100%, full)
    # I did mention this is approximate, right?
    pbat=math.floor((vbat-2.7)/3.3*100)
    # It's possible to get some pretty wild readings
    # so let's fake normality if that happens
    if pbat<0:
      pbat=0
    if pbat>100:
      pbat=100
    return pbat

def keywait(limit):
  # Wait for LIMIT seconds whilst obeying button presses
  nowtime=time.time()
  global row, laststartjoke, firstrun
  while time.time() < nowtime+limit:
    if badger.pressed(badger2040.BUTTON_UP):
      # Move to previous joke
      row=laststartjoke-3
      if row<0:
        row=math.floor(len(lines)/3)*3
      return('d')
    if badger.pressed(badger2040.BUTTON_DOWN):
      # Move to next joke
      row=laststartjoke+3
      if row>len(lines):
        row=0
      return('u')
    if badger.pressed(badger2040.BUTTON_A) and not firstrun:
      # Move to random joke
      # Do not do this immediately after boot (not firstrun)
      # because A may be used to wake the device
      row=math.floor(random.randint(0,len(lines)-1)/3)*3
      laststartjoke=row
      return('a')
    if badger.pressed(badger2040.BUTTON_B):
      # Display a static image then end the program
      # e-paper / e-ink holds its image after power off
      # so we'll show a nice "Merry Christmas" image
      my_image = bytearray(int(296 * 128 / 8))
      open("merry-christmas.bin", "r").readinto(my_image)
      # Be more thorough about clearing the screen
      # We want to minimise ghosting if the device
      # will be off for a long time
      badger.update_speed(badger2040.UPDATE_NORMAL)
      badger.pen(15)
      badger.clear()
      badger.update()
      # Now display the static image
      badger.image(my_image)
      badger.update_speed(badger2040.UPDATE_FAST)
      badger.update()
      sys.exit()
    if badger.pressed(badger2040.BUTTON_C):
      # Show the battery percentage, for what it's worth
      leftjustify("Battery: "+str(get_battery_level())+"%")
      time.sleep(3)
      return('')
    # Wait for 10 milliseconds
    time.sleep(0.01)
    # It's now safe to read button A
    firstrun=False
  return ''

def leftjustify(instr):
  # Display a string on the screen
  # Left-justify it; put in line breaks
  # to prevent it running off the right edge
  outword=''
  outline=''
  numlines=1
  badger.pen(backgroundcolour)
  badger.clear()
  badger.pen(foregroundcolour)
  # Build up words separated by spaces
  # Once a word is found, consider whether adding it to the
  # current line, will go past the right edge of the screen
  # Write each line to the screen
  for s in instr:
    if s==' ':
      if badger.measure_text(outline+' '+outword)+(originx*2)>width:
        badger.text(outline,originx,originy+(numlines*heightfactor))
        numlines=numlines+1
        outline=outword
        outword=''
      else:
        if len(outline)>0:
          outline=outline+' '+outword
        else:
          outline=outword
        outword=''
    else:
      outword=outword+s
  # There's no more text, but we still need to
  # output the most recent word
  if badger.measure_text(outline+' '+outword)+(originx*2)>width:
    badger.text(outline,originx,originy+(numlines*heightfactor))
    numlines=numlines+1
    outline=outword
  else:
    if len(outline)>0:
      outline=outline+' '+outword
    else:
      outline=outword
  badger.text(outline,originx,originy+(numlines*heightfactor))
  badger.update()
  return

# cracker-jokes.txt is a text file
# jokes are on two lines (call and response)
# with a blank line between each joke
# e.g.
## Doctor, doctor, I feel like a pair of curtains
## Well, pull yourself together
##
# Note that only base ASCII characters are supported
# Beware of fancy single quotes - replace with ' apostrophe
inputfile=open('cracker-jokes.txt','r')
lines=inputfile.readlines()
inputfile.close()
if len(lines)<3:
  leftjustify("Not enough in cracker-jokes.txt")
  sys.exit()
# Randomise our starting joke
row=math.floor(random.randint(0,len(lines)-1)/3)*3

# Loop through the jokes in sequence
while True:
  # Remember where we parked
  # (note position of 1st line of joke)
  laststartjoke=row

  # Display 1st part of joke (setup)
  leftjustify(lines[row].strip())

  # Wait and obey button presses
  if keywait(callsecs)!='':
    # A button was pressed which requires us
    # to skip the punchline
    continue
  row=row+1
  if row>=len(lines):
    # If we've gone past the end, loop to start
    # This shouldn't happen on 1st part, but just in case
    row=0

  # Display 2nd line of joke (punchline)
  leftjustify(lines[row].strip())

  # Wait and obey button presses
  if keywait(responsesecs)!='':
    # A button was pressed which requires us
    # to skip the punchline
    continue
  row=row+2
  # If we've gone past the end, loop to start
  if row>=len(lines):
    row=0
